#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include "error.h"
#include "packetio.h"

#define MAX_HEADER_SIZE 2
#define MAX_FRAGMENTS 255

enum FrameType {
    Single,
    Fragmented,
};

typedef struct Reassembly {
    uint64_t write_index;
    uint8_t data[];
} Reassembly_t;

typedef struct PacketIO {
    uint64_t mtu;
    uint64_t max_fragment_data;

    uint8_t n_fragments;
    uint8_t fragment_index;
    Reassembly_t *reassembly;
    packet_receive_callback packet_receive_callback;
    void* packet_receive_user_data;
    fragment_receive_init fragment_receive_init;
    fragment_receive_callback fragment_receive_callback;
    void* fragment_receive_user_data;

    uint8_t send_n_fragments;
    uint8_t send_fragment_index;
    uint8_t *send_buf;
    uint64_t send_buf_index;
    int send_first_fragment;
    frame_send_callback frame_send_callback;
    void* frame_send_user_data;
} PacketIO_t;

void reassemble_callback_init(size_t max_len, void *user_data) {
    PacketIO_t *packet_io = (PacketIO_t*)user_data;
    free(packet_io->reassembly);
    packet_io->reassembly = malloc(sizeof(Reassembly_t) + max_len);
    if (packet_io->reassembly == NULL) {
        error("packet io: no more memory for reassembly\n");
    }
    packet_io->reassembly->write_index = 0;
}

void reassemble_callback_receive(uint8_t *fragment, size_t len, int last, void *user_data) {
    PacketIO_t *packet_io = (PacketIO_t*)user_data;
    if (fragment == NULL) {
        // receive error
        free(packet_io->reassembly);
        packet_io->reassembly = NULL;
        return;
    }
    memcpy(packet_io->reassembly->data + packet_io->reassembly->write_index, fragment, len);
    packet_io->reassembly->write_index += len;
    if (last) {
        packet_io->packet_receive_callback(packet_io->reassembly->data, packet_io->reassembly->write_index, packet_io->packet_receive_user_data);
        free(packet_io->reassembly);
        packet_io->reassembly = NULL;
    }
}

void receive_frame_single(uint8_t *frame, size_t len, PacketIO_t *packet_io) {
    packet_io->fragment_receive_init(len, packet_io->fragment_receive_user_data);
    packet_io->fragment_receive_callback(frame + 1, len - 1, 1, packet_io->fragment_receive_user_data);
}

void receive_frame_fragmented(uint8_t *frame, size_t len, PacketIO_t *packet_io) {
    if (len < 2) {
        return;
    }

    if (packet_io->n_fragments == 0) {
        packet_io->n_fragments = frame[1];
        packet_io->fragment_receive_init(packet_io->n_fragments * packet_io->max_fragment_data, packet_io->fragment_receive_user_data);
        packet_io->fragment_index = 0;
        packet_io->fragment_receive_callback(frame + 2, len - 2, 0, packet_io->fragment_receive_user_data);
    } else {
        if (packet_io->fragment_index != frame[1]) {
            packet_io->fragment_receive_callback(NULL, 0, 1, packet_io->fragment_receive_user_data);
            packet_io->n_fragments = 0;
            return;
        }
        int last = 0;
        if ((packet_io->fragment_index + 2) >= (packet_io->n_fragments)) {
            last = 1;
            packet_io->n_fragments = 0;
        }
        packet_io->fragment_receive_callback(frame + 2, len - 2, last, packet_io->fragment_receive_user_data);
        packet_io->fragment_index++;
    }
}

void packetio_frame_receive(uint8_t *frame, size_t len, PacketIO_t *packet_io) {
    if (frame[0] == Single) {
        receive_frame_single(frame, len, packet_io);
    } else if (frame[0] == Fragmented) {
        receive_frame_fragmented(frame, len, packet_io);
    }
}

int packetio_fragment_init(size_t max_len, PacketIO_t *packet_io) {
    packet_io->send_fragment_index = 0;
    packet_io->send_buf_index = 0;
    packet_io->send_first_fragment = 0;
    if (max_len > MAX_FRAGMENTS * packet_io->max_fragment_data) {
        packet_io->send_n_fragments = 0;
        return 1;
    }
    packet_io->send_n_fragments = (max_len - 1) / packet_io->max_fragment_data + 1;
    return 0;
}

void send_fragment_single(uint8_t *fragment, size_t len, PacketIO_t *packet_io) {
    if (fragment == NULL) {
        return;
    }
    packet_io->send_buf[0] = Single;
    memcpy(packet_io->send_buf + 1, fragment, len);
    packet_io->frame_send_callback(packet_io->send_buf, len + 1, packet_io->frame_send_user_data);
    packet_io->send_n_fragments = 0;
}

void send_fragment_fragmented(uint8_t *fragment, size_t len, PacketIO_t *packet_io) {
    if (fragment == NULL) {
        packet_io->send_buf[0] = Fragmented;
        if (packet_io->send_first_fragment == 0) {
            send_fragment_single(packet_io->send_buf + 2, len, packet_io);
        } else {
            packet_io->send_buf[1] = packet_io->send_fragment_index++;
        }
        packet_io->frame_send_callback(packet_io->send_buf, packet_io->send_buf_index + 2, packet_io->frame_send_user_data);
        packet_io->send_n_fragments = 0;
        return;
    }

    while (len != 0) {
        uint64_t copy_len = MIN(len, packet_io->max_fragment_data - packet_io->send_buf_index);
        memcpy(packet_io->send_buf + packet_io->send_buf_index + 2, fragment, copy_len);
        packet_io->send_buf_index += copy_len;
        fragment += copy_len;
        len -= copy_len;

        if (packet_io->send_buf_index == packet_io->max_fragment_data) {
            packet_io->send_buf[0] = Fragmented;
            if (packet_io->send_first_fragment == 0) {
                packet_io->send_buf[1] = packet_io->send_n_fragments;
                packet_io->send_first_fragment = 1;
            } else {
                packet_io->send_buf[1] = packet_io->send_fragment_index++;
            }
            packet_io->frame_send_callback(packet_io->send_buf, packet_io->max_fragment_data + 2, packet_io->frame_send_user_data);
            packet_io->send_buf_index = 0;
        }

        if ((packet_io->send_fragment_index + 1) == packet_io->send_n_fragments) {
            packet_io->send_n_fragments = 0;
            return;
        }
    }
}

void packetio_fragment_send(uint8_t *fragment, size_t len, PacketIO_t *packet_io) {
    if (packet_io->send_n_fragments == 0) {
        return;
    }
    if (packet_io->send_n_fragments == 1) {
        assert(len <= packet_io->max_fragment_data);
        send_fragment_single(fragment, len, packet_io);
        return;
    }
    send_fragment_fragmented(fragment, len, packet_io);
}

void packetio_packet_send(uint8_t *packet, size_t len, PacketIO_t *packet_io) {
    packetio_fragment_init(len, packet_io);
    packetio_fragment_send(packet, len, packet_io);
    packetio_fragment_send(NULL, 0, packet_io);
}

PacketIO_t *packetio_init(
        uint64_t mtu,
        packet_receive_callback packet_receive_callback,
        void *packet_receive_user_data,
        frame_send_callback frame_send_callback,
        void *frame_send_user_data) {

    PacketIO_t *packet_io = malloc(sizeof(PacketIO_t));
    if (packet_io == NULL) {
        return NULL;
    }
    bzero(packet_io, sizeof(PacketIO_t));
    packet_io->mtu = mtu;
    packet_io->max_fragment_data = mtu - MAX_HEADER_SIZE;
    packet_io->send_buf = malloc(packet_io->mtu);
    if (packet_io->send_buf == NULL) {
        free(packet_io);
        return NULL;
    }
    packet_io->packet_receive_callback = packet_receive_callback;
    packet_io->packet_receive_user_data = packet_receive_user_data;
    packet_io->frame_send_callback = frame_send_callback;
    packet_io->frame_send_user_data = frame_send_user_data;

    packet_io->fragment_receive_init = reassemble_callback_init;
    packet_io->fragment_receive_callback = reassemble_callback_receive;
    packet_io->fragment_receive_user_data = packet_io;

    return packet_io;
}

PacketIO_t *packetio_init_streaming(
        uint64_t mtu,
        fragment_receive_init fragment_receive_init,
        fragment_receive_callback fragment_receive_callback,
        void *fragment_receive_user_data,
        frame_send_callback frame_send_callback,
        void *frame_send_user_data) {

    PacketIO_t *packet_io = malloc(sizeof(PacketIO_t));
    if (packet_io == NULL) {
        return NULL;
    }
    bzero(packet_io, sizeof(PacketIO_t));
    packet_io->mtu = mtu;
    packet_io->max_fragment_data = mtu - MAX_HEADER_SIZE;
    packet_io->send_buf = malloc(packet_io->mtu);
    if (packet_io->send_buf == NULL) {
        free(packet_io);
        return NULL;
    }
    packet_io->fragment_receive_init = fragment_receive_init;
    packet_io->fragment_receive_callback = fragment_receive_callback;
    packet_io->fragment_receive_user_data = fragment_receive_user_data;
    packet_io->frame_send_callback = frame_send_callback;
    packet_io->frame_send_user_data = frame_send_user_data;

    return packet_io;
}
