#include <arpa/inet.h>
#include <errno.h>
#include <netinet/in.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <unistd.h>
#include "com.h"
#include "error.h"
#include "packetio.h"

#define MAX_FILES 256

enum Commands {
    Create,
    Read,
    Write,
    List,
    Delete,
    Uncompress,
    ReadUncompressed,
    DeleteUncompressed,
    Exit,
};

enum ResponseStatus {
    Success,
    NoMemory,
    InvalidLength,
    NoFreeFile,
    NoSuchFile,
    FileTooSmall,
    BackendSuccess = 100,
    BackendNoMemory,
    BackendNoFreeFile,
};

struct File {
    size_t size;
    uint8_t *data;
};

struct MainState {
    Com_t *com_frontend;
    Com_t *com_backend;
    PacketIO_t *packet;
    struct File compressed[MAX_FILES];
    struct File uncompressed[MAX_FILES];
    pthread_t backend_thread;
};

struct File *find_free_file(struct File *files, uint8_t *index) {
    for (int i = 0; i < MAX_FILES; i++) {
        if (files[i].size == 0) {
            if (index != NULL) {
                *index = i;
            }
            return files + i;
        }
    }
    return NULL;
}

int frontend_receive_frame(uint8_t *msg, size_t len, void *user_data) {
    struct MainState *main_state = (struct MainState*)user_data;
    uint8_t error = Success;
    uint8_t response_buf[MAX_FRAME_LEN];
    uint8_t response_len = 1;

    struct File *file;
    uint16_t offset;
    uint64_t read_len;

    switch (msg[0]) {
        case Create:
            if (len != 3) {
                error = InvalidLength;
                break;
            }
            file = find_free_file(main_state->compressed, response_buf + 1);
            if (file == NULL) {
                error = NoFreeFile;
                break;
            }
            file->size = msg[1] + msg[2] * 0x100;
            file->data = calloc(file->size, sizeof(uint8_t));
            if (file->data == NULL) {
                file->size = 0;
                error = NoMemory;
                break;
            }
            response_len = 2;
            break;
        case Read:
            if (len != 4) {
                error = InvalidLength;
                break;
            }
            file = main_state->compressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            offset = msg[2] + msg[3] * 0x100;
            if (offset > file->size) {
                error = FileTooSmall;
                break;
            }
            read_len = MIN(file->size - offset, MAX_FRAME_LEN - 1);
            memcpy(response_buf + 1, file->data + offset, read_len);
            response_len = read_len + 1;
            break;
        case Write:
            if (len < 4) {
                error = InvalidLength;
                break;
            }
            file = main_state->compressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            offset = msg[2] + msg[3] * 0x100;
            if (offset + len - 4 > file->size) {
                error = FileTooSmall;
                break;
            }
            memcpy(file->data + offset, msg + 4, len - 4);
            break;
        case List:
            if (len != 1) {
                error = InvalidLength;
                break;
            }
            response_len = 2 + sizeof(size_t);
            for (uint64_t i = 0; i < MAX_FILES; i++) {
                file = main_state->compressed + i;
                if (file->size != 0) {
                    response_buf[0] = 0;
                    response_buf[1] = i;
                    *(size_t*)(response_buf + 2) = file->size;
                    com_send(response_buf, response_len, main_state->com_frontend);
                }
            }
            for (uint64_t i = 0; i < MAX_FILES; i++) {
                file = main_state->uncompressed + i;
                if (file->size != 0) {
                    response_buf[0] = 1;
                    response_buf[1] = i;
                    *(size_t*)(response_buf + 2) = file->size;
                    com_send(response_buf, response_len, main_state->com_frontend);
                }
            }
            response_len = 1;
            break;
        case Delete:
            if (len != 2) {
                error = InvalidLength;
                break;
            }
            file = main_state->compressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            free(file->data);
            file->data = NULL;
            file->size = 0;
            break;
        case Uncompress:
            if (len != 2) {
                error = InvalidLength;
                break;
            }
            file = main_state->compressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            packetio_packet_send(file->data, file->size, main_state->packet);
            break;
        case ReadUncompressed:
            if (len != 4) {
                error = InvalidLength;
                break;
            }
            file = main_state->uncompressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            offset = msg[2] + msg[3] * 0x100;
            if (offset > file->size) {
                error = FileTooSmall;
                break;
            }
            read_len = MIN(file->size, MAX_FRAME_LEN - 1);
            memcpy(response_buf + 1, file->data + offset, read_len);
            response_len = read_len + 1;
            break;
        case DeleteUncompressed:
            if (len != 2) {
                error = InvalidLength;
                break;
            }
            file = main_state->uncompressed + msg[1];
            if (file->size == 0) {
                error = NoSuchFile;
                break;
            }
            free(file->data);
            file->data = NULL;
            file->size = 0;
            break;
        case Exit:
            return 1;
    }
    response_buf[0] = error;
    com_send(response_buf, response_len, main_state->com_frontend);
    return 0;
}

void backend_send_frame(uint8_t *frame, size_t len, void *user_data) {
    struct MainState *main_state = (struct MainState*)user_data;
    com_send(frame, len, main_state->com_backend);
}

void backend_receive_packet(uint8_t *packet, size_t len, void *user_data) {
    struct MainState *main_state = (struct MainState*)user_data;
    uint8_t file_index;
    struct File *file = find_free_file(main_state->uncompressed, &file_index);
    uint8_t error = BackendSuccess;
    uint8_t response_buf[MAX_FRAME_LEN];
    uint8_t response_len = 1;

    if (file == NULL) {
        error = BackendNoFreeFile;
    } else {
        file->data = malloc(len);
        if (file->data == NULL) {
            error = BackendNoMemory;
        } else {
            memcpy(file->data, packet, len);
            file->size = len;
            response_buf[1] = file_index;
            *(size_t*)(response_buf + 2) = len;
            response_len = 2 + sizeof(size_t);
        }
    }

    response_buf[0] = error;
    com_send(response_buf, response_len, main_state->com_frontend);
}

int backend_receive_frame(uint8_t *frame, size_t len, void *user_data) {
    struct MainState *main_state = (struct MainState*)user_data;
    packetio_frame_receive(frame, len, main_state->packet);
    return 0;
}

void *backend_receive_thread(void *arg) {
    struct MainState *main_state = (struct MainState*)arg;
    com_receive_loop(main_state->com_backend);
    return NULL;
}

int main(int argc, char *argv[]) {
    struct MainState *main_state = calloc(1, sizeof(struct MainState));
    struct sockaddr_in addr;

    int backend_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (backend_fd == -1) {
        error("main: socket: %s\n", strerror(errno));
    }

    bzero(&addr, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = inet_addr("127.0.0.1");
    addr.sin_port = htons(9090);

    if (connect(backend_fd, (struct sockaddr*) &addr, sizeof(addr)) != 0) {
        error("main: connect: %s\n", strerror(errno));
    }

    main_state->com_frontend = com_init(STDIN_FILENO, STDOUT_FILENO, frontend_receive_frame, main_state);
    if (main_state->com_frontend == NULL) {
        error("main: error initializing frontend communication\n");
    }
    main_state->com_backend = com_init(backend_fd, backend_fd, backend_receive_frame, main_state);
    if (main_state->com_backend == NULL) {
        error("main: error initializing backend communication\n");
    }
    main_state->packet = packetio_init(MAX_FRAME_LEN, backend_receive_packet, main_state, backend_send_frame, main_state);
    if (main_state->packet == NULL) {
        error("main: error initializing packet layer\n");
    }

    if (pthread_create(&main_state->backend_thread, NULL, backend_receive_thread, main_state) != 0) {
        error("main: error initializing backend thread\n");
    }

    com_receive_loop(main_state->com_frontend);
    close(backend_fd);
}
