#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "com.h"
#include "error.h"

typedef struct Com {
    int receive_fd;
    int send_fd;
    frame_receive_callback receive_callback;
    void* user_data;
} Com_t;

static void readn(int fd, uint8_t *buf, size_t n) {
    while (n > 0) {
        ssize_t n_read = read(fd, buf, n);
        if (n_read <= 0) {
            error("com: read fd: %s\n", strerror(errno));
            exit(-1);
        }
        n -= n_read;
        buf += n_read;
    }
}

static void writen(int fd, uint8_t *buf, size_t n) {
    while (n > 0) {
        ssize_t n_write = write(fd, buf, n);
        if (n_write <= 0) {
            error("com: write fd: %s\n", strerror(errno));
            exit(-1);
        }
        n -= n_write;
        buf += n_write;
    }
}

Com_t *com_init(int receive_fd, int send_fd, frame_receive_callback receive_callback, void *user_data) {
    Com_t *com = malloc(sizeof(Com_t));
    if (com == NULL) {
        return NULL;
    }
    com->receive_fd = receive_fd;
    com->send_fd = send_fd;
    com->receive_callback = receive_callback;
    com->user_data = user_data;
    return com;
}

void com_receive_loop(Com_t *com) {
    uint8_t size;
    uint8_t frame[MAX_FRAME_LEN];
    int stop = 0;

    while (!stop) {
        readn(com->receive_fd, &size, 1);
        if (size == 0) {
            continue;
        }
        readn(com->receive_fd, frame, size);
        stop = com->receive_callback(frame, size, com->user_data);
    };
}


void com_send(uint8_t *frame, size_t n, Com_t *com) {
    assert(n <= MAX_FRAME_LEN);
    writen(com->send_fd, (uint8_t*)&n, 1);
    writen(com->send_fd, frame, n);
}
