use smithay::wayland::{
    compositor::with_states, foreign_toplevel_list::ForeignToplevelHandle,
    shell::xdg::XdgToplevelSurfaceData, shm::BufferData,
};
use wayland_server::protocol::wl_surface::WlSurface;

pub trait BufferHelperExt {
    fn as_slice(&self, ptr: *const u8) -> &'static [u8];
    fn as_slice_mut(&self, ptr: *mut u8) -> &'static mut [u8];
}

pub trait WlSurfaceExt {
    fn get_title(&self) -> Option<String>;
    fn get_app_id(&self) -> Option<String>;
    fn get_handle(&self) -> Option<ForeignToplevelHandle>;
}

impl WlSurfaceExt for WlSurface {
    fn get_title(&self) -> Option<String> {
        with_states(self, |states| {
            states
                .data_map
                .get::<XdgToplevelSurfaceData>()?
                .lock()
                .ok()?
                .title
                .clone()
        })
    }

    fn get_app_id(&self) -> Option<String> {
        with_states(self, |states| {
            states
                .data_map
                .get::<XdgToplevelSurfaceData>()?
                .lock()
                .ok()?
                .app_id
                .clone()
        })
    }

    fn get_handle(&self) -> Option<ForeignToplevelHandle> {
        with_states(self, |states| states.data_map.get().cloned())
    }
}

impl BufferHelperExt for BufferData {
    fn as_slice(&self, ptr: *const u8) -> &'static [u8] {
        unsafe {
            core::slice::from_raw_parts(
                ptr.offset(self.offset as isize),
                (self.stride * self.height) as usize,
            )
        }
    }

    fn as_slice_mut(&self, ptr: *mut u8) -> &'static mut [u8] {
        unsafe {
            core::slice::from_raw_parts_mut(
                ptr.offset(self.offset as isize),
                (self.stride * self.height) as usize,
            )
        }
    }
}
