use std::{
    fs::Permissions,
    os::unix::fs::PermissionsExt,
    sync::Arc,
    thread,
    time::{Duration, Instant},
};

use anyhow::Result;

use smithay::wayland::compositor::CompositorClientState;
use wayland_server::{
    backend::{ClientData, ClientId, DisconnectReason},
    Display, ListeningSocket,
};

mod app;
mod ext_image_capture_source;
mod ext_image_copy_capture;
mod utils;
use app::App;

fn main() -> Result<()> {
    let mut display: Display<App> = Display::new()?;
    let dh = display.handle();

    let mut state = App::new(&dh);
    let socket_path: std::path::PathBuf = std::env::var("WAYLAND_DISPLAY")?.into();
    let listener = ListeningSocket::bind_absolute(socket_path.clone())?;
    std::fs::set_permissions(socket_path, Permissions::from_mode(0o777))?;

    loop {
        let next = Instant::now() + Duration::from_millis(20);

        state.send_frame_callbacks();

        if let Some(stream) = listener.accept()? {
            display
                .handle()
                .insert_client(stream, Arc::new(ClientState::default()))?;
        }

        display.dispatch_clients(&mut state)?;
        display.flush_clients()?;

        thread::sleep(next - Instant::now());
    }
}

#[derive(Default)]
struct ClientState(CompositorClientState);

impl ClientData for ClientState {
    fn initialized(&self, _client_id: ClientId) {}
    fn disconnected(&self, _client_id: ClientId, _reason: DisconnectReason) {}
}
