use std::sync::{Arc, Mutex};

use smithay::wayland::{
    compositor::with_states,
    shell::xdg::ToplevelSurface,
    shm::{with_buffer_contents, with_buffer_contents_mut, BufferData},
};
use wayland_protocols::ext::image_copy_capture::v1::server::{
    ext_image_copy_capture_frame_v1::{ExtImageCopyCaptureFrameV1, Request as FrameRequest},
    ext_image_copy_capture_manager_v1::{ExtImageCopyCaptureManagerV1, Request as ManagerRequest},
    ext_image_copy_capture_session_v1::{ExtImageCopyCaptureSessionV1, Request as SessionRequest},
};
use wayland_server::{protocol::wl_buffer::WlBuffer, Dispatch, GlobalDispatch, Resource};

use crate::{app::App, utils::BufferHelperExt};

struct FrameDataInner {
    src: Option<WlBuffer>,
    dst: Option<WlBuffer>,
    buffer_data: BufferData,
}
struct FrameData(Arc<Mutex<FrameDataInner>>, ExtImageCopyCaptureSessionV1);

impl FrameData {
    fn new(session: ExtImageCopyCaptureSessionV1) -> Self {
        Self(
            Arc::new(Mutex::new(FrameDataInner {
                src: None,
                dst: None,
                buffer_data: BufferData {
                    offset: 0,
                    width: 0,
                    height: 0,
                    stride: 0,
                    format: wayland_server::protocol::wl_shm::Format::Argb8888,
                },
            })),
            session,
        )
    }

    fn src(&self) -> Option<WlBuffer> {
        self.0.lock().unwrap().src.clone()
    }

    fn set_src(&self, src: WlBuffer) {
        self.0.lock().unwrap().src = Some(src);
    }

    fn dst(&self) -> Option<WlBuffer> {
        self.0.lock().unwrap().dst.clone()
    }

    fn set_dst(&self, dst: WlBuffer) {
        self.0.lock().unwrap().dst = Some(dst);
    }

    fn data(&self) -> BufferData {
        self.0.lock().unwrap().buffer_data
    }

    fn set_data(&self, data: BufferData) {
        self.0.lock().unwrap().buffer_data = data;
    }

    fn session(&self) -> &ExtImageCopyCaptureSessionV1 {
        &self.1
    }
}

impl Dispatch<ExtImageCopyCaptureFrameV1, FrameData> for App {
    fn request(
        _state: &mut Self,
        _client: &wayland_server::Client,
        resource: &ExtImageCopyCaptureFrameV1,
        request: FrameRequest,
        data: &FrameData,
        _dhandle: &wayland_server::DisplayHandle,
        _data_init: &mut wayland_server::DataInit<'_, Self>,
    ) {
        match request {
            FrameRequest::Destroy => {
                let Some(session) = data.session().data::<CopyCaptureSession>() else {
                    return;
                };
                session.remove_frame(resource);
            }
            FrameRequest::AttachBuffer { buffer } => {
                data.set_data(with_buffer_contents(&buffer, |_, _, data| data).unwrap());
                data.set_dst(buffer);
            }
            FrameRequest::DamageBuffer { .. } => {}
            FrameRequest::Capture => {
                let dst_buffer = data.dst().unwrap();
                let src_buffer = data.src().unwrap();
                let buffer_data = data.data();
                let src = with_buffer_contents(&src_buffer, |src, _, _| {
                    buffer_data.as_slice(src).to_owned()
                })
                .unwrap();
                with_buffer_contents_mut(&dst_buffer, |dst, _, _| {
                    buffer_data.as_slice_mut(dst).copy_from_slice(&src);
                })
                .unwrap();

                resource.ready();
            }
            _ => unreachable!(),
        }
    }
}

struct CopyCaptureSession(Arc<Mutex<CopyCaptureSessionInner>>);

impl CopyCaptureSession {
    fn new(src: ToplevelSurface) -> Self {
        Self(Arc::new(Mutex::new(CopyCaptureSessionInner {
            src,
            frames: vec![],
        })))
    }

    fn update_buffer(&self, buffer: WlBuffer) {
        self.0.lock().unwrap().frames.iter().for_each(|frame| {
            let Some(data) = frame.data::<FrameData>() else {
                return;
            };

            let buffer_data = with_buffer_contents(&buffer, |_, _, data| data).unwrap();
            data.set_src(buffer.clone());
            data.set_data(buffer_data);
        });
    }

    fn add_frame(&self, frame: ExtImageCopyCaptureFrameV1) {
        self.0.lock().unwrap().frames.push(frame);
    }

    fn remove_frame(&self, frame: &ExtImageCopyCaptureFrameV1) {
        self.0.lock().unwrap().frames.retain(|f| f != frame);
    }

    fn src(&self) -> ToplevelSurface {
        self.0.lock().unwrap().src.clone()
    }
}

struct CopyCaptureSessionInner {
    src: ToplevelSurface,
    frames: Vec<ExtImageCopyCaptureFrameV1>,
}

impl Dispatch<ExtImageCopyCaptureSessionV1, CopyCaptureSession> for App {
    fn request(
        _state: &mut Self,
        _client: &wayland_server::Client,
        resource: &ExtImageCopyCaptureSessionV1,
        request: SessionRequest,
        data: &CopyCaptureSession,
        _dhandle: &wayland_server::DisplayHandle,
        data_init: &mut wayland_server::DataInit<'_, Self>,
    ) {
        match request {
            SessionRequest::CreateFrame { frame } => {
                data.add_frame(data_init.init(frame, FrameData::new(resource.clone())));
            }
            SessionRequest::Destroy => {
                with_states(data.src().wl_surface(), |states| {
                    let Some(sessions) = states.data_map.get::<Arc<Mutex<Vec<SessionHandle>>>>()
                    else {
                        return;
                    };
                    sessions
                        .lock()
                        .unwrap()
                        .retain(|session| &**session != resource);
                });
            }
            _ => unreachable!(),
        }
    }
}

pub(crate) struct SessionHandle(ExtImageCopyCaptureSessionV1);

impl core::ops::Deref for SessionHandle {
    type Target = ExtImageCopyCaptureSessionV1;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl SessionHandle {
    pub fn update_buffer(&self, buffer: WlBuffer) {
        let data = with_buffer_contents(&buffer, |_, _, data| data).unwrap();
        self.buffer_size(data.width as u32, data.height as u32);
        self.done();
        let Some(session) = self.data::<CopyCaptureSession>() else {
            return;
        };
        session.update_buffer(buffer);
    }
}

impl Dispatch<ExtImageCopyCaptureManagerV1, ()> for App {
    fn request(
        _state: &mut Self,
        _client: &wayland_server::Client,
        resource: &ExtImageCopyCaptureManagerV1,
        request: ManagerRequest,
        _data: &(),
        _dhandle: &wayland_server::DisplayHandle,
        data_init: &mut wayland_server::DataInit<'_, Self>,
    ) {
        match request {
            ManagerRequest::CreateSession {
                session, source, ..
            } => {
                let Some(source) = source.data::<ToplevelSurface>() else {
                    resource.post_error(0u32, "internal error");
                    return;
                };
                let session = data_init.init(session, CopyCaptureSession::new(source.clone()));
                with_states(source.wl_surface(), |states| {
                    states
                        .data_map
                        .get_or_insert(|| Arc::new(Mutex::new(vec![])))
                        .lock()
                        .unwrap()
                        .push(SessionHandle(session.clone()));
                });
                session.shm_format(wayland_server::protocol::wl_shm::Format::Argb8888);
                session.shm_format(wayland_server::protocol::wl_shm::Format::Xrgb8888);
                session.done();
            }
            ManagerRequest::CreatePointerCursorSession { .. } => {
                resource.post_error(0u32, "unsupported")
            }
            ManagerRequest::Destroy => {}
            _ => unreachable!(),
        };
    }
}

impl GlobalDispatch<ExtImageCopyCaptureManagerV1, ()> for App {
    fn bind(
        _state: &mut Self,
        _handle: &wayland_server::DisplayHandle,
        _client: &wayland_server::Client,
        resource: wayland_server::New<ExtImageCopyCaptureManagerV1>,
        _global_data: &(),
        data_init: &mut wayland_server::DataInit<'_, Self>,
    ) {
        data_init.init(resource, ());
    }
}
