<?php

class DB {
    private $conn;

    public function __construct() {
        $this->conn = new PDO('sqlite:/tmp/db.sqlite');
        $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }

    public function exec($sql) {
        error_log($sql);
        return $this->conn->exec($sql);
    }

    public function query($sql) {
        error_log($sql);
        return $this->conn->query($sql);
    }

    public function delete($table, $where = []) {
        if (!has_perms("{$table}_delete")) {
            show_error("missing perm: {$table}_delete");
            return;
        }
        $sql = 'DELETE FROM ' . $this->quoteName($table);
        $sql .= $this->buildWhere($where);
        return $this->exec($sql);
    }

    public function insert($table, $values) {
        if (!has_perms("{$table}_add")) {
            show_error("missing perm: {$table}_add");
            return;
        }
        $sql = 'INSERT INTO ' . $this->quoteName($table) . ' (';
        $sql .= implode(', ', array_map(array($this, 'quoteName'), array_keys($values)));
        $sql .= ') VALUES (';
        $sql .= implode(', ', array_map(array($this, 'quoteValue'), array_values($values)));
        $sql .= ') RETURNING *';
        return $this->query($sql)->fetch();
    }

    public function update($table, $query) {
        if (!has_perms("{$table}_edit")) {
            show_error("missing perm: {$table}_edit");
            return;
        }
        $sql = 'UPDATE ' . $this->quoteName($table) . ' SET';
        foreach ($query['SET'] as $name => $value) {
            $sql .= ' ' . $this->quoteName($name) . ' = ' . $this->quoteValue($value);
        }
        if (isset($query['WHERE'])) $sql .= $this->buildWhere($query['WHERE']);
        return $this->exec($sql);
    }

    public function select($q) {
        $sql = 'SELECT ';
        $sql .= $q['SELECT'] === '*' ? '*' : implode(', ', array_map(array($this, 'quoteName'), $q['SELECT']));
        if (isset($q['FROM'])) $sql .= ' FROM ' . $this->quoteName($q['FROM']);
        if (isset($q['WHERE'])) $sql .= $this->buildWhere($q['WHERE']);
        return $this->query($sql)->fetchAll() ?? [];
    }

    private function buildWhere($where, $op = 'AND') {
        $sql = '';
        foreach ($where as $name => $value) {
            if (!empty($sql)) {
                $sql .= " $op ";
            }
            if (($name === 'OR') || ($name === 'AND')) {
                $sql .= ' (' . $this->buildWhere($value, $name) . ')';
            } else if ($name === 'NOT') {
                $sql .= ' NOT (' . $this->buildWhere($value) . ')';
            } else {
                $sql .= ' ' . $this->quoteName($name) . $this->buildTerm($value);
            }
        }
        return ' WHERE' . $sql;
    }

    private function buildTerm($term) {
        if (is_array($term)) {
            if (count($term) == 2 && isset($term[0]) && $this->isOperator($term[0])) {
                $comparison = $term[0];
                $criterion_value = $term[1];
            } else {
                return 'IN ' . $this->buildValue($term);
            }
        } else {
            $comparison = '=';
            $criterion_value = $term;
        }
        return " $comparison " . $this->buildValue($criterion_value);
    }

    private function isOperator($operator) {
        return in_array($operator, [
            '=', '!=', '<', '<=', '>', '>=', '<>',
            'LIKE', 'NOT LIKE', 'IN', 'NOT IN',
        ], true);
    }

    private function buildValue($value) {
        if (is_array($value)) {
            foreach ($value as $k => $v) {
                $value[$k] = $this->quoteValue($v);
            }
            return '(' . implode(', ', $value) . ')';
        }
        return $this->quoteValue($value);
    }

    private function quoteName($name) {
        return "`$name`";
    }

    private function quoteValue($value) {
        return $this->conn->quote($value);
    }
}
